%
% FICHERO PARA ILUSTRACION DE LOS DISTINTOS TIPOS DE MATRICES QUE APARECEN
% EN DIFERENCIAS FINITAS.
%

%%
% caso unidimensional
%

% recordamos problema y''-5y'+10y=10x con cc y(0)=0, y(1)=100
% funcion fd_solucion_directa

clear; clc;

a = 0.0;       % limite inferior del dominio
b = 1.0;       % limites superior del dominio
               % CONDICIONES DE CONTORNO
ya = 0.0;      % valor de la funcion en el limite inferior del dominio
yb = 1.0e2;    % valor de la funcion en el limite superior del dominio

n = 17; 
[~,y,errrel] = fd_solucion_directa(n);

% copiamos a continuacion la parte del fichero que construye la matriz
x = linspace( a, b, n); 
dx = x(2)-x(1);

% coeficientes de la ODE discretizada
c1 = 1.0 / (2.0 - 10.0*dx^2);
c2 = c1 * ( 1.0 - 2.5*dx);
c3 = c1 * ( 1.0 + 2.5*dx);
c4 = -c1*10.0*dx^2;

% matriz del problema lineal (DIMENSION N-2)
%
% -c3*y(ii-1) + y( ii ) - c2*y(ii+1) = c4*x( ii );
%

dd = ones(1, n-2);
ds = ones(1, n-3)*(-c2);
di = ones(1, n-3)*(-c3);
b  = c4*x( 2:n-1 ); 
b(1) = b(1) + c3*ya;
b(end) = b(end) + c2*yb;
b = b';

Mfd = diag(dd,0) + diag(ds,1) + diag(di,-1);
figure(1); imagesc(Mfd); 
figure(2); spy(Mfd); 


%%
% MATRICES PARA PROBLEMAS DE DIFERENCIAS FINITAS BIDIMENSIONALES
% ecuacion de laplace en una L ( o un cuadrado, cambiando el argumento de 
% la funcion numgrid). CONDICIONES DE DIRICHLET.
%
%          
%                             CN
%        |    ___________________________________ XNE
%        |    |                                  |
%        |    |                                  |
%        |    |                                  |
%        |    |                                  |
%        |    |                                  |
%        |    |                                  |
%        | CW |                                  |CE
%        |    |                                  |
%        |    |                                  |
%        |    |                                  |
%        |    |                                  |
%        |    |                                  |
%        |    |__________________________________|
%        |   XSW              CS
%        |
%        |-------------------------------------------> columna
%        V
%      fila
%   
%      
%                             CN                
%        |     __________________________________ XNW
%        |    |                                  |
%        |    |                                  |
%        | CW1|                                  |
%        |    |                                  |
%        |    |                                  |
%        |    |________________                  |CE
%        |           CS1       |                 |
%        |    |                |                 |
%        |                 CW2 |                 |
%        |    |                |                 |
%        |                     |                 |
%        |    X _   _   _   _  |_________________|
%        |   (0,0)                     CS2
%        |   XSW
%        |-------------------------------------------> columna
%        V
%      fila

clear; clc;

% parametros fisicos
xsw = [  0.0  0.0];
xne = [ 10.0 10.0];
u0n = 1.0;
u0w = 1.0;
u0s = 0.0;
u0e = 0.0;

% parametros numericos
m = 51; 

dx = xne - xsw;
dx = dx./(m-1);

%L = numgrid('L',m); 
L = numgrid('S',m);  % CASO CUADRADO
nL = length(find(L)); % numero de puntos de la malla igual al numero
                      % de elementos no nulos de la matriz L

% definimos una malla ampliada con contornos numerados

[Lo, ino, cs, cn, ce, cw] = malla_orlada(L);
no = length(ino);

A = delsq(Lo)/dx(1)^2; 
figure(1); clf; spy(A);
figure(2); clf; imagesc(A);

AR = A( 1:nL, 1:nL);
%figure(1); clf; imagesc(AR);
AE = A( 1:nL, (nL+1):(nL+no) );
%figure(2); clf; imagesc(AE);

u = zeros( nL + no, 1);
u( cw ) =   u0w;
u( cn ) =   u0n;
u( cs ) =   u0s;
u( ce ) =   u0e;

% para calcular B, lo hacemos con u nulo salvo en elementos del contorno
B = -[AR AE]*u;

u(1:nL) = AR\B;

% ahora hay que pasar el array de nuevo a la disposicion de la malla
uu = zeros(size(Lo));
uu( Lo > 0 ) = u( Lo(Lo>0) );
uu = uu(2:end-1,2:end-1);
% damos a las esquinas un valor ilusorio
uu(1 ,1    ) = 0.5*( u0n + u0w);
uu(1 ,end)   = 0.5*( u0n + u0e);
uu(end, 1)   = 0.5*( u0s + u0w);
uu(end, end) = 0.5*( u0s + u0e);

[ X1, X2] = meshgrid( xsw(1):dx(1):xne(1), ...
                      xne(2):(-dx(2)):xsw(2) );
 
figure(3); clf;
surf( X1, X2, uu);
view(76, 46);

%%
% MATRICES PARA PROBLEMAS DE DIFERENCIAS FINITAS BIDIMENSIONALES
% ecuacion de laplace en una L ( o un cuadrado, cambiando el argumento de 
% la funcion numgrid). CONDICIONES DE NEUMANN.
%
% RESOLVEMOS SOLO EL CASO DEL CUADRADO (para evitar sutilezas con el rincon
% presente en el contorno de la L).
%          
%                             CN
%        |    ___________________________________ XNE
%        |    |                                  |
%        |    |                                  |
%        |    |                                  |
%        |    |                                  |
%        |    |                                  |
%        |    |                                  |
%        | CW |                                  |CE
%        |    |                                  |
%        |    |                                  |
%        |    |                                  |
%        |    |                                  |
%        |    |                                  |
%        |    |__________________________________|
%        |   XSW              CS
%        |
%        |-------------------------------------------> columna
%        V
%      fila
%   

clear; clc;

% parametros fisicos
xsw = [  0.0  0.0];
xne = [ 10.0 10.0];


% parametros numericos
m = 21; 

dx = xne - xsw;
dx = dx./(m-1);

%L = numgrid('L',m); 
L = numgrid('S',m);  % CASO CUADRADO
nL = length(find(L)); % numero de puntos de la malla

% definimos una malla ampliada con contornos numerados

[Lo, ino, cs, cn, ce, cw] = malla_orlada(L);
no = length(ino);
% para condiciones de Dirichlet, incluimos las esquinas
% Para no complicar, lo hacemos solo para el dominio cuadrado

nesq = 4; % numero de esquinas
esq = max(ino)+1:max(ino)+nesq;
Lo(2, 2)    = esq(1);     % NW
Lo(2,end-1) = esq(2);     % NE
Lo(end-1,2) = esq(3);     % SW
Lo(end-1,end-1) = esq(4); % SE

A = delsq(Lo)/dx(1)^2; 
figure(1); clf; spy(A);
figure(2); clf; imagesc(A);

% AR = A( 1:nL, 1:nL);
% %figure(1); clf; imagesc(AR);
% AE = A( 1:nL, (nL+1):(nL+no) );
% %figure(2); clf; imagesc(AE);

% condiciones de contorno para la derivada normal
% derivadas normales (hacia el interior) en N S E W
cc = [ 1.0 2.0 0.0 -3.0];
%cc = [ 1.0 -1.0 0.0 0.0];
%cc = [ 0 0 1.0 -1.0];
cc(1:2) = cc(1:2)/(xne(1)-xsw(1));
cc(3:4) = cc(3:4)/(xne(2)-xsw(2));

A( (nL+1):(nL+no), 1:nL ) = 2*A( (nL+1):(nL+no), 1:nL );

B = zeros(nL+no+nesq,1);
B(cn)  = B(cn) - 2*cc(1)/dx(1);
B(cs)  = B(cs) - 2*cc(2)/dx(1);
B(ce)  = B(ce) - 2*cc(3)/dx(1);
B(cw)  = B(cw) - 2*cc(4)/dx(1);

% hay que corregir los nodos de las esquinas para que contribuyan
% con la derivada en la direccion del contorno
% ESTO DEPENDE DEL DOMINIO ELEGIDO. NO SE PUEDE GENERALIZAR A OTROS 
% DOMINIOS
A( nL+no+1:nL+no+nesq, 1:nL+no) = 2*A( nL+no+1:nL+no+nesq, 1:nL+no);
% esquina NW
B(esq(1)) = B(esq(1)) - 2*(cc(1)+cc(4))/dx(1);
B(esq(2)) = B(esq(2)) - 2*(cc(1)+cc(3))/dx(1);
B(esq(3)) = B(esq(3)) - 2*(cc(2)+cc(4))/dx(1);
B(esq(4)) = B(esq(4)) - 2*(cc(2)+cc(3))/dx(1);

u = A\B;

% ahora hay que pasar el array de nuevo a la disposicion de la malla
uu = zeros(size(Lo));
uu( Lo > 0 ) = u( Lo(Lo>0) );
uu = uu(2:end-1,2:end-1);

[ X1, X2] = meshgrid( xsw(1):dx(1):xne(1), ...
                      xne(2):(-dx(2)):xsw(2) );
 
figure(3); clf;
surf( X1, X2, uu);
view(76, 46);

[u1, u2] = gradient( uu, dx(1), (-dx(2)));
                  
figure(2);clf;
quiver( X1, X2, u1, u2); hold on; contour(X1, X2, uu);

% dibujamos valores de las derivadas normales en contornos
% Estas derivadas normales NO SON LAS MISMAS QUE LAS EXIGIDAS,
% sino las evaluadas en puntos intermedios entre los nodos de la 
% malla.
figure(4);clf;
plot( u1(:,1),'ro'); hold on; plot(u1(:,end),'go'); hold on; 
plot(u2(1,:),'r*') ; hold on; plot(u2(end,:),'b*')
